/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F4011"  Board */
/* Hardware    : ET-BASE dsPIC30F4011                 */
/* Target MCU  : dsPIC30F4011                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Interface Character LCD      */
/*             : 4 Bit Interface on RB0..RB5          */	
/******************************************************/
/* Character 16x2 4-Bit Mode Interface                */
/* D4 = RB0						  				      */ 
/* D5 = RB1						   				      */
/* D6 = RB2						   				      */
/* D7 = RB3						   				      */
/* RS = RB4						   				      */
/* EN = RB5										      */
/******************************************************/

#include <p30f4011.h>                                	// dsPIC30F4011 MPU Register

/* Setup Configuration For ET-BASE dsPIC30F4011 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);							// Disable Clock Switching,Enable Fail-Salf Clock
                                                        // Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);											// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					// Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);									// Code Protect OFF
/* End Configuration For ET-BASE dsPIC30F4011 */

// Character LCD Interface Pins 
#define TRIS_DATA_PIN_4     TRISBbits.TRISB0			// Direction D4
#define TRIS_DATA_PIN_5     TRISBbits.TRISB1			// Direction D5
#define TRIS_DATA_PIN_6     TRISBbits.TRISB2			// Direction D6
#define TRIS_DATA_PIN_7     TRISBbits.TRISB3			// Direction D7
#define TRIS_RS             TRISBbits.TRISB4   			// Direction RS
#define TRIS_E              TRISBbits.TRISB5 			// Direction E

#define DATA_PIN_4          LATBbits.LATB0				// RB0 = D4 LCD
#define DATA_PIN_5          LATBbits.LATB1				// RB1 = D5 LCD
#define DATA_PIN_6          LATBbits.LATB2				// RB2 = D6 LCD
#define DATA_PIN_7          LATBbits.LATB3				// RB3 = D7 LCD
#define RS_PIN              LATBbits.LATB4     			// RB4 = RS LCD
#define E_PIN               LATBbits.LATB5     			// RB5 = E  LCD
								
/* Display ON/OFF Control */
#define DON                 0x0F  						// Display on      
#define DOFF                0x0B  						// Display off     
#define CURSOR_ON           0x0F  						// Cursor on       
#define CURSOR_OFF          0x0D  						// Cursor off      
#define BLINK_ON            0x0F  						// Cursor Blink    
#define BLINK_OFF           0x0E  						// Cursor No Blink 

/* Cursor or Display Shift */
#define SHIFT_CUR_LEFT      0x13  						// Cursor shifts to the left   
#define SHIFT_CUR_RIGHT     0x17  						// Cursor shifts to the right  
#define SHIFT_DISP_LEFT     0x1B  						// Display shifts to the left  
#define SHIFT_DISP_RIGHT    0x1F  						// Display shifts to the right 

/* Function Prototypes */
void Initial_4bitLCD(void);								// Initial LCD Interface
void SetCGRamAddr(unsigned char);						
void SetDDRamAddr(unsigned char);						// Set Cursor Address
void WriteCmdLCD(unsigned char);						// Write Command
void WriteDataLCD(unsigned char);						// Write Data
void PutsLCD(unsigned char*);							// Print Message
void Delay_tW_LCD(void);								// Enable Pulse Delay	
void Busy_LCD(void);									// Wait LCD Busy
void Delay(unsigned long int);							// Delay Time Function

int main(void)
{  
  Initial_4bitLCD();									// Initial LCD 4 Bit Interface
 
  // Loop Print Message to LCD16 x 2 //
  while(1)	  											// Loop Continue
  {
    SetDDRamAddr(0x00);									// Start Cursor Line-1   
    PutsLCD((unsigned char *)"ET-dsPIC 30F4011");
    SetDDRamAddr(0x40);									// Start Cursor Line-2   
    PutsLCD((unsigned char *)"Run 117.9648 MHz");
    Delay(5000000);										// Delay Display

    SetDDRamAddr(0x00);									// Start Cursor Line-1    
    PutsLCD((unsigned char *)"16 Bit / 29 MIPS");
    SetDDRamAddr(0x40);									// Start Cursor Line-2   
    PutsLCD((unsigned char *)"BY...ETT CO.,LTD");
    Delay(5000000);										// Delay Display      
   }  

}	  

void Initial_4bitLCD(void)
{
  TRIS_DATA_PIN_4 = 0;									// Set Port Direction = Output
  TRIS_DATA_PIN_5 = 0;
  TRIS_DATA_PIN_6 = 0;
  TRIS_DATA_PIN_7 = 0;
  TRIS_RS = 0;
  TRIS_E = 0;

  RS_PIN = 0;											// Instruction Select
  E_PIN = 0;											// Disable Strobe
  Delay(50000);											// Power-ON Delay (minimum of 15ms) 
  
  WriteCmdLCD(0x33); 									// Initial 4 Bit Interface Format  
  WriteCmdLCD(0x32);									
  WriteCmdLCD(0x28);									// Function Set (DL=0 4-Bit,N=1 2 Line,F=0 5X7)							

  WriteCmdLCD(0x0C);									// Display on/off Control (Entry Display,Cursor off,Cursor not Blink)
  WriteCmdLCD(0x06);  									// Entry Mode Set (I/D=1 Increment,S=0 Cursor Shift)	  
  WriteCmdLCD(0x01);    								// Clear Display  (Clear Display,Set DD RAM Address=0) 
}

void SetCGRamAddr(unsigned char address)
{
  RS_PIN = 0;  											// Select Instruction  
  DATA_PIN_7 = 0;										// DB7 Must be 0 For Setting CGRam ADDR
  DATA_PIN_6 = 1;										// DB6 Must be 1 For Setting CGRam ADDR  
  DATA_PIN_5 = ((address & 0x20)>>5);
  DATA_PIN_4 = ((address & 0x10)>>4);    
  E_PIN = 1;											// Strobe High Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();

  RS_PIN = 0;  											// Select Instruction
  DATA_PIN_7 = ((address & 0x08)>>3);
  DATA_PIN_6 = ((address & 0x04)>>2);
  DATA_PIN_5 = ((address & 0x02)>>1);
  DATA_PIN_4 = ((address & 0x01)>>0);
  E_PIN = 1;											// Strobe Low Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;   
  Busy_LCD();  
}

void SetDDRamAddr(unsigned char address)
{  
  RS_PIN = 0;   										// Select Instruction 
  DATA_PIN_7 = 1;										// DB7 Must be 1 For Setting DDRam ADDR 
  DATA_PIN_6 = ((address & 0x40)>>6); 
  DATA_PIN_5 = ((address & 0x20)>>5);
  DATA_PIN_4 = ((address & 0x10)>>4);
  E_PIN = 1;											// Strobe High Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
 
  RS_PIN = 0;   										// Select Instruction 
  DATA_PIN_7 = ((address & 0x08)>>3);
  DATA_PIN_6 = ((address & 0x04)>>2);
  DATA_PIN_5 = ((address & 0x02)>>1);
  DATA_PIN_4 = ((address & 0x01)>>0);
  E_PIN = 1;    										// Strobe Low Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN =0;
  Busy_LCD();
} 

void WriteCmdLCD(unsigned char cmd)
{
  RS_PIN = 0;   										// Select Instruction 
  DATA_PIN_7 = ((cmd & 0x80)>>7);
  DATA_PIN_6 = ((cmd & 0x40)>>6);
  DATA_PIN_5 = ((cmd & 0x20)>>5);
  DATA_PIN_4 = ((cmd & 0x10)>>4);
  E_PIN = 1;											// Strobe High Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
  
  RS_PIN = 0;   										// Select Instruction 
  DATA_PIN_7 = ((cmd & 0x08)>>3);
  DATA_PIN_6 = ((cmd & 0x04)>>2);
  DATA_PIN_5 = ((cmd & 0x02)>>1);
  DATA_PIN_4 = (cmd & 0x01);
  E_PIN = 1;											// Strobe Low Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
  Delay(50000);											// 1.64mS Delay
}  

void WriteDataLCD(unsigned char data)
{
  RS_PIN = 1;  											// Select Data                    
  DATA_PIN_7 = ((data & 0x80)>>7);
  DATA_PIN_6 = ((data & 0x40)>>6);
  DATA_PIN_5 = ((data & 0x20)>>5);
  DATA_PIN_4 = ((data & 0x10)>>4);
  E_PIN = 1;											// Strobe High Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();

  RS_PIN = 1;  											// Select Data   
  DATA_PIN_7 = ((data & 0x08)>>3);
  DATA_PIN_6 = ((data & 0x04)>>2);
  DATA_PIN_5 = ((data & 0x02)>>1);
  DATA_PIN_4 = (data & 0x01);
  E_PIN = 1;											// Strobe Low Nibble
  Delay_tW_LCD();										// Enable Pulse Delay
  E_PIN = 0;
  Busy_LCD();
}  

void PutsLCD(unsigned char* buffer)
{
  while(*buffer != '\0')								// Loop Until End of Message
  {    
    WriteDataLCD(*buffer); 								// Write Character to LCD
    buffer++;
  } 
}

void Delay_tW_LCD(void)   								// Enable Pulse Delay
{
  int i;
  for(i=0;i<18;i++);       
}

void Busy_LCD(void)										// 40uS Delay
{
  unsigned int i;
  for(i=0;i<1800;i++);      
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void Delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}							// Loop Decrease Counter	
}

